// ===== MODE ARTISTIQUE - DIAGRAMME CIRCULAIRE AVEC BARRES RADIALES =====

let artisticSvg = null;
let currentArtisticYear = '2019';

function initArtisticD3() {
    console.log('Initialisation du mode artistique D3.js');
    
    // Nettoyer l'ancien SVG si existant
    if (artisticSvg) {
        d3.select('#artistic-svg').remove();
        artisticSvg = null;
    }
    
    const container = d3.select('#artistic-d3-container');
    if (container.empty()) {
        console.error('Conteneur artistique non trouvé');
        return;
    }
    
    // Dimensions
    const containerNode = container.node();
    const width = containerNode.clientWidth;
    const height = containerNode.clientHeight;
    const centerX = width / 2;
    const centerY = height / 2;
    const radius = Math.min(width, height) / 2 - 100;
    
    // Créer le SVG
    artisticSvg = container.append('svg')
        .attr('id', 'artistic-svg')
        .attr('width', width)
        .attr('height', height);
    
    // Groupe principal
    const g = artisticSvg.append('g')
        .attr('transform', `translate(${centerX}, ${centerY})`);
    
    // Récupérer les données
    const data = getArtisticRadialData(currentArtisticYear);
    
    if (!data || data.length === 0) {
        console.warn('Pas de données pour le mode artistique');
        g.append('text')
            .attr('text-anchor', 'middle')
            .attr('fill', '#fff')
            .attr('font-size', '24px')
            .text('Aucune donnée disponible');
        return;
    }
    
    console.log('Données chargées:', data);
    
    // Créer le diagramme radial
    createRadialBarChart(g, data, radius);
}

function getArtisticRadialData(year) {
    const yearsData = window.CSV_DATA?.years;
    if (!yearsData) return [];
    
    const tvData = yearsData.tv?.[year];
    const radioData = yearsData.radio?.[year];
    
    const data = [];
    
    // Ajouter les chaînes TV (limiter à 8)
    if (tvData) {
        const tvChannels = Object.entries(tvData)
            .filter(([key, value]) => typeof value === 'number')
            .slice(0, 8);
        
        tvChannels.forEach(([channel, value]) => {
            data.push({
                label: channel,
                value: value,
                type: 'tv',
                color: '#ff6b8b'
            });
        });
    }
    
    // Ajouter les chaînes Radio (limiter à 8)
    if (radioData) {
        const radioChannels = Object.entries(radioData)
            .filter(([key, value]) => typeof value === 'number')
            .slice(0, 8);
        
        radioChannels.forEach(([channel, value]) => {
            data.push({
                label: channel,
                value: value,
                type: 'radio',
                color: '#6bccb2'
            });
        });
    }
    
    return data;
}

function createRadialBarChart(g, data, maxRadius) {
    const innerRadius = 80;
    const barWidth = (2 * Math.PI) / data.length;
    
    // Échelle pour la longueur des barres
    const radiusScale = d3.scaleLinear()
        .domain([0, 100])
        .range([innerRadius, maxRadius]);
    
    // Cercle intérieur (fond)
    g.append('circle')
        .attr('r', innerRadius)
        .attr('fill', 'rgba(255, 255, 255, 0.05)')
        .attr('stroke', 'rgba(255, 255, 255, 0.2)')
        .attr('stroke-width', 2);
    
    // Cercles de référence (grille)
    const gridLevels = [25, 50, 75, 100];
    gridLevels.forEach(level => {
        g.append('circle')
            .attr('r', radiusScale(level))
            .attr('fill', 'none')
            .attr('stroke', 'rgba(255, 255, 255, 0.1)')
            .attr('stroke-width', 1)
            .attr('stroke-dasharray', '5,5');
        
        // Labels des niveaux
        g.append('text')
            .attr('x', 5)
            .attr('y', -radiusScale(level))
            .attr('fill', 'rgba(255, 255, 255, 0.5)')
            .attr('font-size', '10px')
            .text(level + '%');
    });
    
    // Groupe pour les barres
    const barsGroup = g.append('g').attr('class', 'radial-bars');
    
    // Créer les barres radiales
    data.forEach((d, i) => {
        const angle = i * barWidth - Math.PI / 2;
        const barGroup = barsGroup.append('g')
            .attr('class', 'bar-group')
            .attr('transform', `rotate(${(angle * 180 / Math.PI) + 90})`);
        
        // Barre de fond (gris)
        barGroup.append('rect')
            .attr('x', -barWidth * 40)
            .attr('y', innerRadius)
            .attr('width', barWidth * 80)
            .attr('height', 0)
            .attr('fill', 'rgba(255, 255, 255, 0.1)')
            .attr('rx', 3)
            .transition()
            .duration(1000)
            .delay(i * 50)
            .attr('height', maxRadius - innerRadius);
        
        // Barre de valeur (colorée)
        const bar = barGroup.append('rect')
            .attr('x', -barWidth * 40)
            .attr('y', innerRadius)
            .attr('width', barWidth * 80)
            .attr('height', 0)
            .attr('fill', d.color)
            .attr('rx', 3)
            .attr('opacity', 0.8);
        
        // Animation d'entrée
        bar.transition()
            .duration(1500)
            .delay(i * 50)
            .attr('height', radiusScale(d.value) - innerRadius)
            .attr('opacity', 1);
        
        // Effet de brillance
        barGroup.append('rect')
            .attr('x', -barWidth * 40)
            .attr('y', innerRadius)
            .attr('width', barWidth * 40)
            .attr('height', 0)
            .attr('fill', 'rgba(255, 255, 255, 0.3)')
            .attr('rx', 3)
            .transition()
            .duration(1500)
            .delay(i * 50)
            .attr('height', radiusScale(d.value) - innerRadius);
        
        // Interaction au survol
        barGroup
            .style('cursor', 'pointer')
            .on('mouseenter', function() {
                d3.select(this).select('rect:nth-child(2)')
                    .transition()
                    .duration(200)
                    .attr('opacity', 1)
                    .attr('width', barWidth * 90);
                
                // Afficher le tooltip
                showArtisticTooltip(d, angle);
            })
            .on('mouseleave', function() {
                d3.select(this).select('rect:nth-child(2)')
                    .transition()
                    .duration(200)
                    .attr('opacity', 0.8)
                    .attr('width', barWidth * 80);
                
                hideArtisticTooltip();
            });
    });
    
    // Labels des chaînes (en cercle extérieur)
    const labelsGroup = g.append('g').attr('class', 'radial-labels');
    
    data.forEach((d, i) => {
        const angle = i * barWidth - Math.PI / 2;
        const labelRadius = maxRadius + 30;
        const x = Math.cos(angle) * labelRadius;
        const y = Math.sin(angle) * labelRadius;
        
        const label = labelsGroup.append('text')
            .attr('x', x)
            .attr('y', y)
            .attr('text-anchor', 'middle')
            .attr('dominant-baseline', 'middle')
            .attr('fill', d.color)
            .attr('font-size', '11px')
            .attr('font-weight', '600')
            .attr('opacity', 0)
            .text(d.label);
        
        // Rotation du texte pour qu'il soit lisible
        const rotation = (angle * 180 / Math.PI) + 90;
        if (rotation > 90 && rotation < 270) {
            label.attr('transform', `translate(${x}, ${y}) rotate(${rotation + 180})`);
        } else {
            label.attr('transform', `translate(${x}, ${y}) rotate(${rotation})`);
        }
        
        // Animation d'apparition
        label.transition()
            .duration(1000)
            .delay(i * 50 + 500)
            .attr('opacity', 1);
    });
    
    // Titre central
    const titleGroup = g.append('g').attr('class', 'central-title');
    
    titleGroup.append('text')
        .attr('text-anchor', 'middle')
        .attr('y', -10)
        .attr('fill', '#fff')
        .attr('font-size', '24px')
        .attr('font-weight', '700')
        .attr('opacity', 0)
        .text('Temps de Parole')
        .transition()
        .duration(1000)
        .delay(1000)
        .attr('opacity', 1);
    
    titleGroup.append('text')
        .attr('text-anchor', 'middle')
        .attr('y', 15)
        .attr('fill', '#fff')
        .attr('font-size', '18px')
        .attr('opacity', 0)
        .text('des Femmes')
        .transition()
        .duration(1000)
        .delay(1200)
        .attr('opacity', 1);
    
    titleGroup.append('text')
        .attr('text-anchor', 'middle')
        .attr('y', 40)
        .attr('fill', '#6bccb2')
        .attr('font-size', '32px')
        .attr('font-weight', '700')
        .attr('opacity', 0)
        .text(currentArtisticYear)
        .transition()
        .duration(1000)
        .delay(1400)
        .attr('opacity', 1);
    
    // Légende
    const legend = g.append('g')
        .attr('class', 'legend')
        .attr('transform', `translate(${-maxRadius}, ${maxRadius + 50})`);
    
    legend.append('rect')
        .attr('width', 15)
        .attr('height', 15)
        .attr('fill', '#ff6b8b')
        .attr('rx', 3);
    
    legend.append('text')
        .attr('x', 20)
        .attr('y', 12)
        .attr('fill', '#fff')
        .attr('font-size', '12px')
        .text('Télévision');
    
    legend.append('rect')
        .attr('x', 100)
        .attr('width', 15)
        .attr('height', 15)
        .attr('fill', '#6bccb2')
        .attr('rx', 3);
    
    legend.append('text')
        .attr('x', 120)
        .attr('y', 12)
        .attr('fill', '#fff')
        .attr('font-size', '12px')
        .text('Radio');
}

function showArtisticTooltip(data, angle) {
    const container = d3.select('#artistic-d3-container');
    const width = container.node().clientWidth;
    const height = container.node().clientHeight;
    
    // Supprimer l'ancien tooltip
    d3.select('.artistic-tooltip').remove();
    
    // Créer le tooltip
    const tooltip = container.append('div')
        .attr('class', 'artistic-tooltip')
        .style('position', 'absolute')
        .style('background', 'rgba(0, 0, 0, 0.9)')
        .style('color', '#fff')
        .style('padding', '12px 16px')
        .style('border-radius', '8px')
        .style('border', `2px solid ${data.color}`)
        .style('font-size', '14px')
        .style('pointer-events', 'none')
        .style('z-index', '1000')
        .style('opacity', 0);
    
    tooltip.html(`
        <div style="font-weight: 700; margin-bottom: 5px; color: ${data.color};">${data.label}</div>
        <div style="font-size: 12px; color: #ccc;">${data.type === 'tv' ? 'Télévision' : 'Radio'}</div>
        <div style="font-size: 20px; font-weight: 700; margin-top: 5px;">${data.value.toFixed(1)}%</div>
        <div style="font-size: 11px; color: #999;">Temps de parole femmes</div>
    `);
    
    // Positionner le tooltip
    const x = width / 2 + Math.cos(angle) * 150;
    const y = height / 2 + Math.sin(angle) * 150;
    
    tooltip
        .style('left', x + 'px')
        .style('top', y + 'px')
        .style('transform', 'translate(-50%, -50%)')
        .transition()
        .duration(200)
        .style('opacity', 1);
}

function hideArtisticTooltip() {
    d3.select('.artistic-tooltip')
        .transition()
        .duration(200)
        .style('opacity', 0)
        .remove();
}

function refreshArtisticD3() {
    console.log('Rafraîchissement du mode artistique');
    initArtisticD3();
}

function updateArtisticYear(year) {
    currentArtisticYear = year;
    refreshArtisticD3();
}

// Export des fonctions
window.initArtisticD3 = initArtisticD3;
window.refreshArtisticD3 = refreshArtisticD3;
window.updateArtisticYear = updateArtisticYear;
